<?php
/**
 * WooCommerce MailChimp Discount
 *
 * @package Admin
 * @version 3.8
 */

defined( 'ABSPATH' ) || exit;

class WC_MailChimp_Discount {

  /**
  * Bootstraps the class and hooks required actions & filters.
  *
  */
  protected $options = '';

  public function __construct() {

    $options = get_option( 'wcmd_options' );
    $this->options = get_option( 'wcmd_options' );

    add_filter( 'plugin_action_links_' . WCMD_BASE, array( $this, 'wcmd_settings_link' ) );
    
    add_action( 'admin_enqueue_scripts', array( $this, 'admin_scripts' ) );
    
    add_action( 'plugins_loaded', array($this, 'wcmd_transient_list_ids' ) );
    
    add_action( 'wp_ajax_wcmd_verify_api_key', array( $this, 'wcmd_verify_api_key' ) );
    
    add_action( 'wp_ajax_wcmd_verify_webhook', array( $this, 'wcmd_verify_webhook' ) );
    
    add_action( 'wp_ajax_wcmd_get_mailchimp_lists', array( $this , 'wcmd_fetch_admin_mailchimp_lists' ) );
    
    add_action( 'wp_ajax_wcmd_database_update', array( $this, 'wcmd_database_update' ) );
    
    add_action( 'wcmd_optionsframework_after_validate', array( $this, 'wcmd_save_options_notice' ) );

    if( $this->option( 'wcmd_enabled' ) == 'yes' ) {

      //Add css and js files for the popup
      add_action( 'wp_enqueue_scripts',  array( $this, 'wcmd_enque_scripts' ) );

      //Add shortcode support on the widgets
      add_filter( 'widget_text', 'do_shortcode' );

      //Add shortcode for mailchimp discount.
      add_shortcode( 'wc_mailchimp_discount', array( $this, 'wcmd_render_shortcode' ) );

      add_shortcode( 'wcmd', array( $this, 'wcmd_lang_func' ) );

      //Mailchimp user registration.
      add_action( 'wp_ajax_wcmd_subscribe', array( $this, 'wcmd_subscribe' ) );

      add_action( 'wp_ajax_nopriv_wcmd_subscribe', array( $this, 'wcmd_subscribe' ) );
    
      
      if( $this->option( 'wcmd_disable_popup' ) != 'yes' || $this->option( 'wcmd_btn_trigger' ) == 'yes' )
        add_action( 'wp_footer', array( $this, 'wcmd_display_popup' ) );

      if( $this->option( 'wcmd_restrict' ) == 'yes' && $this->option( 'wcmd_loggedin' ) == 'yes' ) {
        add_filter( 'woocommerce_coupon_is_valid', array( $this,'validate_coupon' ), 10, 2);
      }
      
      // Disable the woocommerce_hold_stock_for_checkout.
      add_filter( 'woocommerce_hold_stock_for_checkout',  array( $this, 'wcmd_hold_stock_for_checkout' ) );
    }
  }

  /**
  * Disable the filter
  * @name wcmd_hold_stock_for_checkout
  */
  public function wcmd_hold_stock_for_checkout() {
   return false;
  }

  /**
  *
  * Check API key
  *
  * @since 3.6
  * @return string
  *
  */
  public function wcmd_save_options_notice() {

    if ( isset( $_POST ) ) {
      $api_key = $_POST['wcmd_options']['wcmd_api_key'];
      if ( empty( $api_key ) ) {
        add_settings_error( 'options-framework', 'save_options', __( 'It seems that still you haven\'t added any mailchimp API key.', 'wc_mailchimp_discount' ), 'error fade in' );
      }
    }

  }

  /**
   * Add new link for the settings under plugin links
   *
   * @param array $links an array of existing links.
   * @return array of links  along with age restricted shopping settings link.
   *
   */
  public function wcmd_settings_link( $links ) {
    $settings_link = '<a href="'.admin_url('admin.php?page=woocommerce-mailchimp-discount').'">Settings</a>';
    array_unshift( $links, $settings_link );
    return $links;
  }

  /**
  *
  * Adds admin style
  *
  * @since 1.0
  * @param empty
  * @return mixed
  *
  */
  public function admin_scripts() {
    $screen = get_current_screen();

    if( $screen->id !== 'toplevel_page_woocommerce-mailchimp-discount' )
      return;
    
    wp_enqueue_style( 'admin-style', plugins_url( 'assets/css/admin-style.css', WCMD_FILE ) );
  }

  /**
  * Adds MailChimp List/Audience into transient
  *
  * @since 3.4
  *
  * @param empty
  * @return Boolean
  *
  */
  public function wcmd_transient_list_ids() {
    //Check transient data exists
    $get_lists = get_transient( 'wcmd_mailchimp_lists' );
    $options = get_option( 'wcmd_options' );
    $api_key = isset( $options['wcmd_api_key'] ) ? trim( $options['wcmd_api_key'] ) : '';

    if( empty( $get_lists ) ) {
      $lists = $this->get_mailchimp_lists( $api_key );

      if( is_array( $lists ) && !empty( $lists ) ) {
        delete_transient( 'wcmd_mailchimp_lists' );
        set_transient( 'wcmd_mailchimp_lists', $lists, 60 * 60 * 24 ); 
      }
    }
  }

  /**
  *
  * Verify MailChimp API Key
  *
  * @since 3.4
  * @param empty
  * @return JSON Object
  *
  */
  public function wcmd_verify_api_key() {

    $apiKey = isset( $_POST['MailChimpAPIKey'] ) ? trim( $_POST['MailChimpAPIKey'] ) : '';
    if( !empty( $apiKey ) ) {
      $apiKey = sanitize_key( $apiKey );
      $mailchimp = new MGMailChimp( $apiKey );
      $check_response = $mailchimp->get('ping');
      if( isset( $check_response['health_status'] ) ) {
        $response = array( 'status' => true, 'msg' => $check_response['health_status'] );
      }
      else {
        $response = array( 'status' => false );
      }
    }
    wp_send_json( $response );
    wp_die();
  }

  /**
  *
  * Makes verify the webhook
  *
  * @since 3.4
  * @param empty
  * @return JSON Object
  *
  */
  public function wcmd_verify_webhook() {
    check_ajax_referer( 'wcmd-verify-webhook-nonce', 'security' );

    $apiKey = isset( $_POST['MailChimpAPIKey'] ) ? trim( $_POST['MailChimpAPIKey'] ) : '';
    $listId = isset( $_POST['MailChimpList'] ) ? trim( $_POST['MailChimpList'] ) : '';
    $webhook_url = isset( $_POST['WebHookUrl'] ) ? trim( $_POST['WebHookUrl'] ) : '';

    if( empty( $apiKey ) || empty( $listId ) ) 
      wp_die();

    $apiKey = sanitize_key( $apiKey );
    $listId = sanitize_key( $listId );

    $mailchimp = new MGMailChimp( $apiKey );
    $get_webhooks = $mailchimp->get( '/lists/'.$listId.'/webhooks' );

    if( is_array( $get_webhooks ) && isset( $get_webhooks['webhooks'] ) ) {
      $added_webhooks = $get_webhooks['webhooks'];

      if( is_array( $added_webhooks ) && !empty( $added_webhooks ) ) {
        $added_webhooks_url = array();
        foreach( $added_webhooks as $added_webhook ) {
          array_push( $added_webhooks_url, $added_webhook['url'] );
        }
      }
      
      if( !in_array( $webhook_url, $added_webhooks_url ) ) {
        // add the webhook here
        $webhook_response = $mailchimp->post( '/lists/'.$listId.'/webhooks',
          array(
            "url"     => $webhook_url,
            "events"  => array(
              'subscribe'     => true,
              'unsubscribe'   => false,
              'profile'       => true,
              'cleaned'       => false,
            ),
            "sources"   => array(
            "user"      => true,
            "admin"     => true,
            "api"       => false
            ),
          )
        );

        if( is_array( $webhook_response ) 
          && isset( $webhook_response['errors'] ) ) {
          $error_messages = '';
          $webhook_errors = $webhook_response['errors'];

          if( is_array( $webhook_errors ) && !empty( $webhook_errors ) ) {
            foreach( $webhook_errors as $key => $webhook_error ) {
              $error_messages .= $webhook_error['message'] . '<br/>';
            }
          }
          $response = array( 'success' => false, 'message_type' => 'error', 'message' => $error_messages);
        }
        else {
          //Webhook added successfully
          $response = array( 'success' => true, 'message_type' => 'success', 'message' => __('Webhook added successfully', 'wc_mailchimp_discount') );
        } 
      }
      else {
        //Webhook already exists
        $response = array( 'success' => false, 'message_type' => 'info', 'message' => __('Webhook already added', 'wc_mailchimp_discount') );
      }
    }
    wp_send_json($response);
    wp_die();
  }

  /**
  * Gets MailChimp lists by MailChimp API and saves them into transient
  *
  * @since 3.4
  *
  * @param empty
  * @return JSON Object
  *
  */
  public function wcmd_fetch_admin_mailchimp_lists() {
    
    check_ajax_referer( 'wcmd-lists-nonce', 'security' );

    $apikey = isset( $_POST['apikey'] ) ? $_POST['apikey'] : '';
    if( empty( $apikey ) )
      return;

    $apikey = sanitize_key( $apikey );

    $get_lists = $this->get_mailchimp_lists( $apikey );

    if( is_array( $get_lists ) && !empty( $get_lists ) ) {
      delete_transient( 'wcmd_mailchimp_lists' );
      set_transient( 'wcmd_mailchimp_lists', $get_lists, 60 * 60 * 24 );
      wp_send_json( $get_lists );
    }
    wp_die();
  }

  /**
  * Output response for database update
  *
  * @param null
  * @return Output response for database update
  *
  */
  public function wcmd_database_update() {
    
    check_ajax_referer( 'wcmd_database_updater', 'security' );

    global $wpdb;
    $options_table = $wpdb->prefix .'options';
    $wcmd_array = array();

    $older_datas = $wpdb->get_results( "SELECT option_name,option_value FROM $options_table WHERE option_name LIKE '%wcmd_%' ");
  
    if( $older_datas ) {
      foreach( $older_datas as $older_data ) {
        if( isset($older_data->option_name) && isset($older_data->option_value) ) {
          $wcmd_array[$older_data->option_name] = $older_data->option_value;
        }
      }
    }

    update_option( 'wcmd_options', $wcmd_array );
    $updated = update_option( 'wcmd_data_imported', 'yes' );
    
    if( $updated ) {
      $response = array( 'status' => 'success' );
    }
    else {
      $response = array( 'status' => 'error' );
    }
    
    wp_send_json( $response );
    exit;
  }

  /**
  * Gets Mailchimp Lists
  *
  * @since 3.4
  *
  * @param string | $api_key
  * @return array 
  *
  */
  private function get_mailchimp_lists($apikey) {

    if( empty( $apikey ) )
      return;

    $mailchimp = new MGMailChimp( $apikey );
    $get_lists = $mailchimp->get('lists', array( 'count'=> '100' ));
    
    if( is_array( $get_lists ) && !empty( $get_lists ) ){
      $lists_array = array();
      foreach( $get_lists as $lists ) {
        if ( is_array( $lists ) ) {
          foreach( $lists as $list ) {
            $lists_array[$list['id']] = $list['name'];
          }
        }
      }
    }
    return $lists_array;
  }

  /**
  *
  * Gets options from the database
  *
  * @since 3.4
  * @param array | $option
  * @return array
  *
  */
  public function option( $option ) {
    if( isset( $this->options[$option] ) && $this->options[$option] != '' )
      return $this->options[$option];
    return '';
  }

  /**
  *
  * Gets language based texts 
  *
  * @since 3.0
  * @param array | $atts 
  * @param string | $content
  * @return mixed
  *
  */
  public function wcmd_lang_func( $atts, $content = "" ) {
    $a = shortcode_atts( array( 'lang' => '' ), $atts );

    if( !function_exists( 'icl_object_id' ) )
      $current_lang = substr( get_bloginfo( 'language' ), 0, 2 );
    else
      $current_lang = ICL_LANGUAGE_CODE;

    if( !empty( $a['lang'] ) && $current_lang == $a['lang'] )
      return $content;
    else
      return;
  }

  /**
  *
  * Adds necessary css and js for the plugin
  *
  * @since 3.0
  * @param empty
  * @return mixed
  *
  */
  public function wcmd_enque_scripts() {

    if( $this->option( 'wcmd_disable_discount' ) != 'yes' && $this->option( 'wcmd_double_optin') == 'yes' && isset( $_POST['type'] ) && $_POST['type'] == 'subscribe' ) {
    
      if( isset( $_POST['data']['merges']['WCMDLANG'] ) && !empty( $_POST['data']['merges']['WCMDLANG'] ) ) {
        $current_lang = $_POST['data']['merges']['WCMDLANG'];
      }
      else {
        if( !function_exists( 'icl_object_id' ) )
          $current_lang = substr( get_bloginfo('language'), 0, 2 );
        else
          $current_lang = ICL_LANGUAGE_CODE;
      }

      $email = $_POST['data']['email'];
      $fname = $_POST['data']['fname'];
      $lname = $_POST['data']['lname'];

      $check_coupon_assign = $this->wcmd_check_assigned_roles( $email );
      if( $check_coupon_assign ) {
        $this->wcmd_send_coupons( $email, $fname, $lname, $current_lang );
      }
      
    }
    
    $overlay_color  = $this->option( 'wcmd_pop_overlay' );
    list($r, $g, $b)  = sscanf($overlay_color, "#%02x%02x%02x");
    $rgb_color      = 'rgba('.$r.','.$g.','.$b.','.$this->option( 'wcmd_overlay_opacity' ).')';
    $height         = $this->option( 'wcmd_popup_height' ) == 0 ? 'auto' : $this->option('wcmd_popup_height' ) . 'px';
    $width          = $this->option( 'wcmd_popup_width' ) == 0 ? 'auto' : $this->option( 'wcmd_popup_width' ) . 'px';
    $bg             = $this->option( 'wcmd_pop_bg' ) ==  '' ? $this->option( 'wcmd_pop_bgcolor') : 'url(' . $this->option( 'wcmd_pop_bg' ) . ')';
    $top_pixel      = $this->option('wcmd_content_top') . 'px';
    $left_pixel     = $this->option('wcmd_content_left') . 'px';
    $form_width     = $this->option( 'wcmd_form_width' ) == 0 ? 'auto' : $this->option( 'wcmd_form_width' ) . 'px';
    $close_color    = $this->option( 'wcmd_close_color' );
    
    if( $close_color == '' )
      $close_color = '#fff';
      
      $css  = '#wcmd_modal{ min-height:' . $height . ';background:' . $bg . ';max-width:' . $width . ';}';
      $css .= '#wcmd_modal .mfp-close{ color:' .$close_color .' !important; }';
      $css .= '#wcmd-form{float:' . $this->option( 'wcmd_form_alignment' ) . '; max-width:' . $form_width . ';}';
      $css .= '.wcmd-title{ color:' . $this->option( 'wcmd_header_color' ) . ';}';
      $css .= '.wcmd_text{ top:' . $top_pixel . ';left:' . $left_pixel . ';}';
      $css .= '.wcmd-btn{ background:' . $this->option( 'wcmd_btn_color' ) . ';color:' . $this->option( 'wcmd_btn_txt_color' ) . ';}';
      $css .= '.wcmd-btn:hover{ background:' . $this->option( 'wcmd_btn_hover' ) . ';}';
      $css .= '#wcmd-form label{ color:' . $this->option( 'wcmd_label_color' ) . ';}';
      $css .= '#wcmd-form .wcmd-confirm{ background:' . $this->option( 'wcmd_checkbox_color' ) . ';}';
      $css .= '.wcmd-form .wcmd-validation.success{ background: '.$this->option( 'wcmd_success_bg_color').'; color: '.$this->option( 'wcmd_success_txt_color').'; border: 1px solid '.$this->option( 'wcmd_success_bg_color').'; }';
      $css .= '.wcmd-form .wcmd-validation.error{ background: '.$this->option('wcmd_error_bg_color').'; color: '.$this->option('wcmd_error_txt_color').'; border: 1px solid '.$this->option('wcmd_error_bg_color').' }';
      $css  .= '#wcmd_modal .wcmd-checkbox-wrap * { color:' . $this->option('wcmd_terms_text_color') . ';}';
      $css  .= 'form .wcmd-fields label { color:' . $this->option('wcmd_terms_text_color') . ';}';
      $css .= '.wcmd_content {display: block;max-width: 500px;margin: 0 auto;}';
      $css .= '.wcmd-form-wrapper form .wcmd-fields input {min-width: auto !important;}';

      //Add animate css
       wp_register_style( 'wcmd-animate-style' , plugins_url( 'woocommerce-mailchimp-discount/assets/css/animate.css' ), array(), '1.0' );

      //Add custombox css
      wp_register_style( 'wcmd-custombox-stylesheet' , plugins_url( 'woocommerce-mailchimp-discount/assets/css/magnific-popup.css' ), array(), '1.0' );

      wp_enqueue_style( 'wcmd-animate-style' );
      wp_enqueue_style( 'wcmd-custombox-stylesheet' );

      //Add our customized css
      wp_add_inline_style( 'wcmd-custombox-stylesheet', $css );

      //Magnific Popup js script
      wp_register_script( 'wcmd-custombox', plugins_url( 'woocommerce-mailchimp-discount/assets/js/jquery.magnific-popup.min.js' ), array( 'jquery' ), '1.0' );

      wp_enqueue_script( 'wcmd-custombox' );

      wp_register_script( 'wcmd-custom', plugins_url( 'woocommerce-mailchimp-discount/assets/js/wcmd-custom.js' ), array( 'jquery' ), '1.0' );
      wp_enqueue_script( 'wcmd-custom' );

      $invalid_email_error = $this->option( 'wcmd_email_error' ) !== '' ? $this->option( 'wcmd_email_error' ) : __('Please enter a valid email id', 'wc_mailchimp_discount');

      $double_optin = $this->option( 'wcmd_double_optin' ) == '1' ? 'yes' : 'no';

      $popup_for_users = $this->show_popup_users() ? 'yes' : 'no';

      wp_localize_script('wcmd-custom', 'wcmd', array(
        'double_optin'          =>  $double_optin,
        'effect'                =>  $this->option( 'wcmd_popup_animation' ),
        'width'                 =>  $this->option( 'wcmd_popup_width' ),
        'overlayColor'          =>  $rgb_color,
        'delay'                 =>  $this->option( 'wcmd_dis_seconds' ),
        'success'               =>  do_shortcode( $this->option('wcmd_success_msg' ) ),
        'cookie_length'         =>  $this->option( 'wcmd_cookie_length' ),
        'wcmd_popup'            =>  $this->option( 'wcmd_disable_popup' ),
        'valid_email'           =>  $invalid_email_error,
        'enable_terms_condition'=>  $this->option( 'wcmd_terms_condition' ),
        'terms_condition_error' =>  $this->option( 'wcmd_terms_condition_error' ),
        'ajax_url'              =>  admin_url( 'admin-ajax.php' ),
        'exit_intent'           =>  $this->option( 'wcmd_exit_intent' ),
        'hinge'                 =>  $this->option( 'wcmd_hinge' ),
        'overlay_click'         =>  $this->option( 'wcmd_overlay_click' ),
        'btn_trigger'           =>  $this->option( 'wcmd_btn_trigger' ),
        'only_btn'              =>  $this->option( 'wcmd_only_btn' ),
        'close_time'            =>  $this->option( 'wcmd_close_seconds' ),
        'wcmd_home'             =>  $this->option( 'wcmd_home' ),
        'disable_popup_on_mobile' => $this->option( 'wcmd_disable_mobile_popup'),
        'is_home'               =>  is_front_page(),
        'signup_redirect'       =>  $this->option( 'wcmd_signup_redirect' ),
        'redirect_url'          =>  $this->option( 'wcmd_redirect_url' ),
        'redirect_timeout'      =>  $this->option( 'wcmd_redirect_timeout' ),
        'email_sent_error'      =>  __( 'There was an error trying to send your message. Please try again later.', 'wc_mailchimp_discount'),
        'popup_for_users'       =>  $popup_for_users,
        'wcmd_subscribe_none'   =>  wp_create_nonce( 'wcmd-subscribe-secret' ),
        'page_to_show_popup'    =>  $this->show_popup_on_pages() ? 'yes' : 'no',
        'product_to_show_popup' =>  $this->show_popup_on_product_page() ? 'yes' : 'no',
        'show_popup_users'      =>  $popup_for_users,
        'already_subscribed'    =>  __('It seems that the email address is already subscribed', 'wc_mailchimp_discount'),
      ));
  }


  /**
  *
  * Checks whom to show the popup
  *
  * @since 3.4
  * @param empty
  * @return boolean
  *
  */
  public function show_popup_users() {
    $popup_users = $this->option( 'wcmd_popup_users' );

    if( $popup_users == 'non_loggedin_users' ) {
      $return = !is_user_logged_in() ? true : false;
    }

    if( $popup_users == 'loggedin_users' ) {
      $return = is_user_logged_in() ? true : false;
    }

    if( $popup_users == 'all' ) {
      $return = true;
    }
    
    return $return;

  }


  /**
  *
  * Checks on which page the popup would be shown
  *
  * @since 3.4
  * @param empty
  * @return boolean
  *
  */
  public function show_popup_on_pages() {

    $popup_selected_pages = $this->option( 'wcmd_popup_enabled_pages' );
    global $post;
    $current_page = $post->post_name;
    $cond = false;

    if( !empty( $popup_selected_pages ) ) {
      $popup_selected_pages = explode( ',', $popup_selected_pages );

      if( is_array( $popup_selected_pages ) && !empty( $popup_selected_pages ) ) {
        if( in_array( $current_page, $popup_selected_pages ) ) {
          $cond = true;
        }
      }
    }

    return $cond;
  }

  /**
  *
  * Checks on which product page the popup would be shown
  *
  * @since 3.6
  * @return boolean
  *
  */
  public function show_popup_on_product_page() {

    $popup_selected_cats = $this->option( 'wcmd_popup_enabled_products' );
    
    global $post;
    $product_id = $post->ID;
    $cond = false;
    
    if( !empty( $popup_selected_cats ) ) {
      $popup_selected_cats = explode( ',', $popup_selected_cats );

      if( !empty( $product_id ) ) {
        
        $cat_ids = wp_get_post_terms( $product_id, 'product_cat' ,array('fields'=>'ids') );

        foreach ( $popup_selected_cats as $catid ) {
          if ( in_array( $catid, $cat_ids ) ) {
            $cond = true;
          }
        }
      }
    }
    
    return $cond;
  }

  /**
  * Output the html for the popup.
  *
  * @param void
  * @return outputs the html for the popup
  *
  */
  public function wcmd_display_popup() {
    $wcmd_title = do_shortcode( $this->option( 'wcmd_pop_header') );
    $fields = $this->option( 'wcmd_fields' );
    $fields_label = $this->option( 'wcmd_form_label' );
    $pop_text = wpautop( stripslashes( $this->option('wcmd_popup_text') ) );
    $pop_text = do_shortcode( $pop_text );
  ?>
  <div id="wcmd_modal" class="mfp-with-anim mfp-hide">
    <?php if( $wcmd_title != '' ) echo '<h4 class="wcmd-title">' . $wcmd_title . '</h4>'; ?>

    <!-- content starts here -->
    <div class="wcmd_content">
      <div class="wcmd-loading"></div>
      
      <div class="wcmd_text">
      <?php 
        $form = '<form class="wcmd-form wcmd_' . $fields . '">';

        $form .= '<div class="wcmd-fields">';
      
        if( $fields == 'email_name' || $fields == 'email_name_all' ) {
          $form .= '<input type="text" aria-label="wcmd_fname" placeholder="'. __( 'Enter first name', 'wc_mailchimp_discount' ) .'" name="wcmd_fname" class="wcmd_fname">';
        }

        if( $fields == 'email_name_all' ) {
          $form .= '<input type="text"  aria-label="wcmd_lname" placeholder="'. __( 'Enter last name', 'wc_mailchimp_discount' ) .'" name="wcmd_lname" class="wcmd_lname">';
        }

        $form .='<input type="text" aria-label="wcmd_email" placeholder="'. __( 'Enter your email', 'wc_mailchimp_discount' ) .'" name="wcmd_email" class="wcmd_email">';


        $form .= '<div class="wcmd-clear"></div>';

        if( class_exists( 'WC_MailChimp_Discount_Extra_Fields' ) ) {
          $options = 'wcmd_options';
          $addon_fields = $this->option( 'wmcd_addon_fields' );
          $form .= apply_filters( 'wmcd_addon_fields_hook', $options, $addon_fields );
          $form .= '<div class="wcmd-clear"></div>';
        }

        $form .= '</div>';


        //checkbox for terms and conditions
        if( $this->option( 'wcmd_terms_condition') == 'yes' ) :
          $term_condition_text = !empty( $this->option('wcmd_terms_condition_text')) ? $this->option('wcmd_terms_condition_text' ) : '';
          $uniq_id = uniqid();

          $form .= '<div class="wcmd-checkbox-wrap">';
          $form .= '<input type="checkbox" id="'.$uniq_id.'" class="wcmd-terms-conditions" name="wcmd_terms_condition">';
          $form .= '<label for='.$uniq_id.'>'.$term_condition_text.'</label>';
          $form .= '</div>';
        endif;

        $form .= '<div class="wcmd-btn-cont">';
        $form .= '<button class="wcmd-btn">' . $this->option( 'wcmd_btn_text' ) . '</button>';
        $form .= '</div><div class="wcmd-clear"></div>';
        $form .= '<div class="wcmd-validation"></div></form>';
        $form .= '<div class="wcmd-clear"></div>';
        
        $form .= '</form>';
        
        //Replace the form code and add the form html.
        echo str_replace( '{WCMD_FORM}', $form, $pop_text );
      ?>
      </div>
    </div>
  </div>
  <?php
  }


  /**
  * Check the subscribed user role has been set for getting coupon
  *
  * @since 3.0
  * @param $email
  * @return outputs boolean
  *
  */
  public function wcmd_check_assigned_roles( $email ) {
    $wcmd_get_roles = $this->option( 'wcmd_roles' );
    $cond = true;
    if( !empty($wcmd_get_roles) ) {
      $wcmd_get_roles = explode( ',', $wcmd_get_roles );
      $get_user = get_user_by( 'email', $email );
      
      if( $get_user ) {
        $user_roles = $get_user->roles;
        if( is_array( $user_roles ) && !empty( $user_roles ) ) {
          $matched_roles = array_intersect( $user_roles, $wcmd_get_roles );
          $matched_roles = count( $matched_roles );

          if( $matched_roles > 0 ) {
            $cond = true;
          }
          else {
            $cond = false;
          }
        }
      }
      else {
        $cond = true;
      }
    }

    return $cond;
  }


  /**
  *
  * Gets all the data related to the coupon assigned
  *
  * @since 3.4
  * @param int | coupon id
  * @return array
  *
  */
  public function wcmd_get_coupons_data($coupon_id) {
    $code   = get_the_title($coupon_id);
    $expire = get_post_meta($coupon_id, 'expiry_date', true);
    // $expire = get_post_meta($coupon_id, 'expiry_date', true);

    $product_ids = get_post_meta($coupon_id, 'product_ids', true);

    if( $product_ids ) {
      $product_ids = explode( ',', $product_ids );

      if ( is_array( $product_ids ) ) {
        foreach ( $product_ids as $product_id ) {
          $product = wc_get_product( $product_id );
          $allowed_products .= '<a href="'.$product->get_permalink().'">'.$product->get_title().'</a>,';
        }
        $allowed_products = rtrim( $allowed_products, ',' );
        $product_ids = implode( ',', $product_ids );
      }
    }

    $exclude_product_ids = get_post_meta( $coupon_id, 'exclude_product_ids', true );

    if( $exclude_product_ids ) {
      $exclude_product_ids = explode( ',', $exclude_product_ids );
      
      if ( is_array( $exclude_product_ids ) ) {
        foreach ( $exclude_product_ids as $product_id ) {
          $product = wc_get_product( $product_id );
          $excluded_products .= '<a href="'.$product->get_permalink().'">'.$product->get_title().'</a>,';
        }
      
        $excluded_products = rtrim( $excluded_products, ',' );
        $exclude_product_ids = implode( ',', $exclude_product_ids );
      }
    }

    $allowed_cats = '';
    $excluded_cats = '';

    $product_categories = get_post_meta( $coupon_id, 'product_categories', array() );


    if ( is_array( $product_categories ) ) {
      foreach ( $product_categories as $k => $cats_id ) {
        if( is_array($cats_id) ) {
          foreach( $cats_id as $cat_id ) {
            $cat = get_term_by( 'id', $cat_id, 'product_cat' );
            $allowed_cats .= '<a href="'.get_term_link( $cat->slug, 'product_cat' ).'">'.$cat->name.'</a>,';
          }
        }
      }
      $allowed_cats = rtrim( $allowed_cats, ',' );
    }
    else
      $product_categories = array();

    $exclude_product_categories = get_post_meta( $coupon_id, 'exclude_product_categories', array() );
    if ( is_array( $exclude_product_categories ) ) {
      foreach ( $exclude_product_categories as $key => $cats_id ) {
        if( is_array( $cats_id ) && !empty( $cats_id ) ) {
          foreach( $cats_id as $cat_id ) {
            $cat = get_term_by( 'id', $cat_id, 'product_cat' );
            $excluded_cats .= '<a href="'.get_term_link( $cat->slug, 'product_cat' ).'">'.$cat->name.'</a>,';
          }
        }
      }
      $excluded_cats = rtrim( $excluded_cats, ',' );
    }
    else
      $exclude_product_categories = array();    

    $coupon_data = array(
        'code'              => $code,
        'expire'            => $expire,
        'allowed_cats'      => $allowed_cats,
        'excluded_cats'     => $excluded_cats,
        'allowed_products'  => $allowed_products,
        'excluded_products' => $excluded_products
      );

    return $coupon_data;
  }


  /**
  *
  * Creates custom copun for the subscribed user
  *
  * @since 3.4
  * @param empty
  * @return mixed
  *
  */
  private function wcmd_create_custom_coupon($email) {
    $code_length = $this->option( 'wcmd_code_length' );

    if( $code_length == '' )
      $code_length = 12;
          
    $prefix = $this->option( 'wcmd_prefix' );
    $code = $prefix . strtoupper( substr( str_shuffle( md5( time() ) ), 0, $code_length ) );
    $type = $this->option( 'wcmd_dis_type' );
    $amount = $this->option( 'wcmd_amount' );
    $product_ids = $this->option( 'wcmd_products' );
    $allowed_products = '';
    $excluded_products = '';
          
    if ( is_array( $product_ids ) ) {
      foreach ( $product_ids as $product_id ) {
        $product = wc_get_product( $product_id );
        $allowed_products .= '<a href="'.$product->get_permalink().'">'.$product->get_title().'</a>,';
      }
      $allowed_products = rtrim( $allowed_products, ',' );
      $product_ids = implode( ',', $product_ids );
    }

    $exclude_product_ids = $this->option( 'wcmd_exclude_products' );
    if ( is_array( $exclude_product_ids ) ) {
      foreach ( $exclude_product_ids as $product_id ) {
        $product = wc_get_product( $product_id );
        $excluded_products .= '<a href="'.$product->get_permalink().'">'.$product->get_title().'</a>,';
      }
      
      $excluded_products = rtrim( $excluded_products, ',' );
      $exclude_product_ids = implode( ',', $exclude_product_ids );
    }

    $product_categories = $this->option( 'wcmd_categories' );
    $allowed_cats = '';
    $excluded_cats = '';
    
    if ( $product_categories ) {
      $product_categories = explode( ',', $product_categories );
      if ( is_array( $product_categories ) ) {
        foreach ( $product_categories as $cat_id ) {
          $cat = get_term_by( 'id', $cat_id, 'product_cat' );
          $allowed_cats .= '<a href="'.get_term_link( $cat->slug, 'product_cat' ).'">'.$cat->name.'</a>,';
        }
        $allowed_cats = rtrim( $allowed_cats, ',' );
      }
  }
    else
      $product_categories = array();

    $exclude_product_categories = $this->option( 'wcmd_exclude_categories' );
    
    if ( $exclude_product_categories ) {
      $exclude_product_categories = explode( ',', $exclude_product_categories );
      if ( is_array( $exclude_product_categories ) ) {
        foreach ( $exclude_product_categories as $cat_id ) {
          $cat = get_term_by( 'id', $cat_id, 'product_cat' );
          $excluded_cats .= '<a href="'.get_term_link( $cat->slug, 'product_cat' ).'">'.$cat->name.'</a>,';
        }
        $excluded_cats = rtrim( $excluded_cats, ',' );
      }
    }
    else
      $exclude_product_categories = array();

    $days = $this->option( 'wcmd_days' );
    $date = '';
    $expire = '';
    $format = $this->option( 'wcmd_date_format' ) == '' ? 'jS F Y' : $this->option( 'wcmd_date_format' );
      
    if ( $days ) {
      $date = date( 'Y-m-d', strtotime( '+'.$days.' days' ) );
      $expire = date_i18n( $format, strtotime( '+'.$days.' days' ) );
    }

    $free_shipping = $this->option( 'wcmd_shipping' );
    $exclude_sale_items = $this->option( 'wcmd_sale' );
    $minimum_amount = $this->option( 'wcmd_min_purchase' );
    $maximum_amount = $this->option( 'wcmd_max_purchase' );
    $customer_email = '';
    
    if ( $this->option( 'wcmd_restrict' ) == 'yes' )
      $customer_email = $email;

      //Add a new coupon when user registers
      $coupon = array(
        'post_title'    => $code,
        'post_content'  => '',
        'post_status'   => 'publish',
        'post_author'   => 1,
        'post_type'     => 'shop_coupon'
      );
      
      $coupon_id = wp_insert_post( $coupon );

      //Add coupon meta data
      update_post_meta( $coupon_id, 'discount_type', $type );
      update_post_meta( $coupon_id, 'coupon_amount', $amount );
      update_post_meta( $coupon_id, 'individual_use', 'yes' );
      update_post_meta( $coupon_id, 'product_ids', $product_ids );
      update_post_meta( $coupon_id, 'exclude_product_ids', $exclude_product_ids );
      update_post_meta( $coupon_id, 'usage_limit', '1' );
      update_post_meta( $coupon_id, 'usage_limit_per_user', '1' );
      update_post_meta( $coupon_id, 'limit_usage_to_x_items', '' );
      update_post_meta( $coupon_id, 'expiry_date', $date );
      update_post_meta( $coupon_id, 'apply_before_tax', 'no' );
      update_post_meta( $coupon_id, 'free_shipping', $free_shipping );
      update_post_meta( $coupon_id, 'exclude_sale_items', $exclude_sale_items );
      update_post_meta( $coupon_id, 'product_categories', $product_categories );
      update_post_meta( $coupon_id, 'exclude_product_categories', $exclude_product_categories );
      update_post_meta( $coupon_id, 'minimum_amount', $minimum_amount );
      update_post_meta( $coupon_id, 'maximum_amount', $maximum_amount );
      update_post_meta( $coupon_id, 'customer_email', $customer_email );
      update_post_meta( $coupon_id, 'woocommerce_mailchimp_discount', 'yes');

      $coupon_data = array(
        'code'    => $code,
        'expire'  => $expire,
        'allowed_cats' => $allowed_cats,
        'excluded_cats' => $excluded_cats,
        'allowed_products' => $allowed_products,
        'excluded_products' => $excluded_products,
        'customer_email' => $customer_email
      );
      return $coupon_data;
  }


  /**
  * Output coupon code.
  *
  * @param $email, $language
  * @return coupon code
  *
  */
  public function wcmd_send_coupons( $email, $fname, $lname, $language ) {
  
    global $woocommerce;
    
    $emails = get_option( 'wcmd_mails', array() );

    $use_predefined_coupon = $this->option( 'wcmd_use_coupons_from' );

    if( $use_predefined_coupon == 'yes' ) {
      $coupon_code = $this->option( 'wcmd_predefined_coupon' );
      $coupon_data = $this->wcmd_get_coupons_data( $coupon_code );
    }
    else {
      $coupon_data = $this->wcmd_create_custom_coupon($email);
    }
    
    $search = array( '{COUPONCODE}', '{COUPONEXPIRY}', '{ALLOWEDCATEGORIES}', '{EXCLUDEDCATEGORIES}', '{ALLOWEDPRODUCTS}', '{EXCLUDEDPRODUCTS}',  '{EmailAddress}', '{FirstName}', '{LastName}' );
    $replace = array( $coupon_data['code'], $coupon_data['expire'], $coupon_data['allowed_cats'], $coupon_data['excluded_cats'], $coupon_data['allowed_products'], $coupon_data['excluded_products'], $email, $fname, $lname );
    $subject = str_replace( $search, $replace, $this->option( 'wcmd_email_sub' ) );
    $subject = do_shortcode( $subject );
    $body = str_replace( $search, $replace, $this->option( 'wcmd_email' ) );
    $body = stripslashes( $body );
    $body = do_shortcode( $body );

    // Change sender name
    add_filter( 'woocommerce_email_from_name', function( $from_name, $wc_email ) {
      if( $wc_email->id == '' ) {
        $from_name = $this->option( 'wcmd_email_name' );
      }
      return $from_name;
    }, 10, 2 );

    // Change sender adress
    add_filter( 'woocommerce_email_from_address', function( $from_email, $wc_email ){

      if( $from_email->id == '' ) {
        $from_email = $this->option( 'wcmd_email_id' );
      }
      return $from_email;
    }, 10, 2 );

    $mailResult = false;
    $mail_response = 'error';

    if ( $this->option( 'wcmd_double_optin' ) == 'yes' ) {
      $headers = "MIME-Version: 1.0" . "\r\n";
      $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
      $headers .= 'From: <"'.$this->option( 'wcmd_email_id' ).'">'.$this->option( 'wcmd_email_name' ).'' . "\r\n";
    }
    else {
      $headers = array('Content-Type: text/html; charset=UTF-8', 'From: '. $this->option( 'wcmd_email_name' ) .' <'. $this->option( 'wcmd_email_id' ) .'>');
    }
    
    $mailer = WC()->mailer();
    $wrapped_message = $mailer->wrap_message($heading, $subject, $body);

    $wc_email = new WC_Email;
    $html_message = $wc_email->style_inline($wrapped_message);
    $mailResult = $mailer->send( $email, $subject, $mailer->wrap_message( $subject, $body ), $headers, '' );

    if( ! $mailResult ) {

      $mail_headers = "MIME-Version: 1.0" . "\r\n";
      $mail_headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
      $mail_headers .= 'From: <"'.$this->option( 'wcmd_email_id' ).'">'.$this->option( 'wcmd_email_name' ).'' . "\r\n";
      $body = htmlspecialchars_decode($body);
      $mailResult = mail($email, $subject, $body, $mail_headers);
    }

    if( $email != $this->option( 'wcmd_test_mail' ) ){
      $emails[] = $email;
      update_option( 'wcmd_mails', $emails );
    }

    if( $mailResult ) 
      $mail_response = 'success';

    return $code_response = array( 'coupon_code' => $coupon_data['code'], 'email_response' => $mail_response );

  }


  /**
  * This function is used to check whether merge field exists or not
  *  If merge fiel not exists then it will create the field
  * @param string $merge_var merge var that would be created.
  * @param string $merge_field merge field name
  * @param boolean $public the field should be public or not.
  *
  */
  public function setup_merge_fields( $field_tag, $field_label, $public, $source_type ) {
  
    if( !empty( $field_label ) && !empty( $field_tag ) ) {

      $field_tag = str_replace(' ', '', $field_tag);
      $field_tag = str_replace('_', '', $field_tag);
      $field_tag = str_replace('-', '', $field_tag);
      $field_tag = substr($field_tag, 0, 10);
      $field_tag = strtoupper($field_tag);

      $options = get_option( 'wcmd_options' );
      $api_key   = !empty( $this->option( 'wcmd_api_key' ) ) ? trim( $this->option( 'wcmd_api_key' ) ) : '';

      $list_id = !empty( $this->option( 'wcmd_list_id' ) ) ? trim( $this->option( 'wcmd_list_id' ) ) : '';

      $MailChimp = new MGMailChimp( $api_key );
      
      $get_merge_vars = $MailChimp->get("/lists/{$list_id}/merge-fields");
        
      if( is_array( $get_merge_vars ) && !empty( $get_merge_vars ) ) {
        $tags_array = array();
        
        foreach( $get_merge_vars as $vars_result ) {
          foreach( $vars_result as $key => $vals ) {
            array_push( $tags_array, $vals['tag'] );
          }
        }

        if( !in_array( $field_tag, $tags_array ) ) {
          $MailChimp->post("/lists/{$list_id}/merge-fields",
            array(
              "tag"           => $field_tag,
              "required"      => false,
              "name"          => $field_label,
              "type"          => $source_type,
              "default_value" => "",
              "public"        => $public,
            )
          );
        }
      }
    }
  }


  /**
  * Assign coupon to the user
  * @param string $email.
  * @param string $language.
  * @param string $disable_discount.
  *
  */
  public function wcmd_assign_coupon( $email, $fname, $lname, $language ) {
    
    $disable_discount = $this->option( 'wcmd_disable_discount' ) == 'yes' ? 'yes' : 'no';

    if( $disable_discount !== 'yes' ) {
      $coupon_response = $this->wcmd_send_coupons( $email, $fname, $lname, $language );
      
      if( isset($coupon_response['email_response']) 
        && $coupon_response['email_response'] == 'success' ) {
        $response['coupon_code'] = $coupon_response['coupon_code'];
        $response['email_response'] = 'success';
        $response['status'] = 'success';
      }
      else {
        $response['email_response'] = 'error';
        $response['status'] = 'error';
      }
    }
    else {
      $response['email_response'] = 'success';
      $response['status'] = 'success';
    }
    return $response;
  }



  /**
  * Lets subscribe the user to our MailChimp by API
  *
  * @param empty
  * @return Output response code for subscription
  *
  */
  public function wcmd_subscribe() {

    check_ajax_referer( 'wcmd-subscribe-secret', 'security' );

    $email  = isset( $_POST['email'] ) ? sanitize_text_field( $_POST['email'] ) : '';
    $fname  = isset( $_POST['fname'] ) ? sanitize_text_field( $_POST['fname'] ) : '';
    $lname  = isset( $_POST['lname'] ) ? sanitize_text_field( $_POST['lname'] ) : '';

    $optin = $this->option( 'wcmd_double_optin' ) == 'yes' ? 'pending' : 'subscribed';

    $source     = $this->option( 'wcmd_source' );

    $merge_vars = array( 'FNAME'=> $fname, 'LNAME'=> $lname );

    $testing_email = $this->option('wcmd_test_mail');

    if( !function_exists('icl_object_id') )
      $current_language = substr(get_bloginfo('language'), 0, 2);
    else
      $current_language = ICL_LANGUAGE_CODE;


    //Sanitize the email
    if( $email == '' )
      return;
    
    $email = sanitize_email( $email );


    //If the email is from test user then don't send them to mailchimp, just send a coupon
    if( !empty($testing_email) && $testing_email == $email ) {
      $response = $this->wcmd_assign_coupon($email, $fname, $lname, $current_language);
      wp_send_json($response);
    }
    else {
      $extrafields = !empty($_POST['extrafields']) ? $_POST['extrafields'] : array();

      if( is_array($extrafields) && !empty($extrafields) ) {
        $addon_merge_fields = apply_filters('wcmd_extra_merge_fields', $extrafields);
      }
      else {
        $addon_merge_fields = array();
      }
      

      $response = $this->wcmd_integrate_mailchimp($email, $fname, $lname, $addon_merge_fields, $current_language);


      if( is_array( $response ) && isset( $response['status']) ) {

        if( $response['type'] == 'success' && $response['status'] == 'success' && !isset($response['title']) ) {

          if( $this->wcmd_check_assigned_roles($email) && $optin !== 'pending' ) {

            $response = $this->wcmd_assign_coupon($email, $fname, $lname, $current_language);
          }
          else {
            $response['email_response'] = 'success';
          }
        }
      }
    }

    wp_send_json($response);
    
    wp_die();
  }


  /**
  *
  * Integrate MailChimp when user tries to subscribe 
  *
  * @param string | $email
  * @param string | $fname
  * @param string | $lname
  * @param array  | $addon_merge_tags
  * @param string | $language
  *
  * @return array with status for subscribe
  *
  */
  public function wcmd_integrate_mailchimp( $email, $fname, $lname, $addon_merge_tags, $language ) {

    $apikey  = !empty( $this->option( 'wcmd_api_key' ) ) ? trim( $this->option( 'wcmd_api_key' ) ) : '';

    $list_id = !empty( $this->option( 'wcmd_list_id' ) ) ? trim( $this->option( 'wcmd_list_id' ) ) : '';

    $optin = $this->option( 'wcmd_double_optin' ) == 'yes' ? 'pending' : 'subscribed';

    $signup_source = $this->option( 'wcmd_source' ) == 'yes' ? 'yes' : 'no';


    if( $signup_source == 'yes' ) {
      
      $source_name = $this->option( 'wcmd_source_link' ) !== '' ? $this->option( 'wcmd_source_link' ) : get_home_url();


      $merge_fields = Array( 'FNAME' => $fname, 'LNAME' => $lname, 'SOURCE' =>  $source_name, 'WCMDLANG' => $language );
      $source_type = 'text';

      $this->setup_merge_fields( 'SOURCE', 'Signup Source', true, $source_type );
    }
    else {
      $merge_fields = Array( 'FNAME' => $fname, 'LNAME' => $lname, 'WCMDLANG' => $language );
    }

    $merge_vars = Array( 'FNAME' => 'First Name', 'LNAME' => 'Last Name', 'SOURCE' =>  'Signup Source', 'WCMDLANG' => 'Language' );


    //Addon Merge Tags
    if( is_array( $addon_merge_tags ) && !empty( $addon_merge_tags ) ) {
      $merge_fields = array_merge( $merge_fields, $addon_merge_tags );
    }

    if( is_array( $merge_vars ) && !empty( $merge_vars ) ) {
      foreach( $merge_vars as $merge_tag => $merge_field ) {
        $this->setup_merge_fields( $merge_tag, $merge_field, true, 'text' );
      }
    }


    if( !empty( $apikey ) && !empty( $list_id ) ) {
      $mailchimp = new MGMailChimp( $apikey );

      $subscriber_hash = $mailchimp->subscriberHash( $email );

      $check_member = $mailchimp->get( "lists/{$list_id}/members/{$subscriber_hash}", [
        'email_address' => $email,
        'status'        => $optin,
        'merge_fields'  => $merge_fields,
        'language'      => $language
      ]);

      if( is_array( $check_member ) && isset( $check_member['status'] ) ) {

        switch ( $check_member['status'] ) {
          case '404':
            $result = $mailchimp->post("lists/{$list_id}/members", [
              'email_address' => $email,
              'status'        => $optin,
              'merge_fields'  => $merge_fields,
            ]);

            if( $result['status'] == '400' ) {
              $result['status'] = 'error';
              $result['type']   = 'error';
              $result['message'] = $result['detail'];
            }
            else {
              $result['status'] = 'success';
              $result['type']   = 'success';
            }
            
          break;

          case 'unsubscribed':
            $result['status'] = 'unsubscribed';
            $result['type']   = 'error';
            $result['message'] = $email . __(' has been unsubcribed before.', 'wc_mailchimp_discount');
          break;

          case 'subscribed':
            $result['status'] = 'subscribed';
            $result['type']   = 'error';
            $result['message'] = $email . __(' is already subscribed before', 'wc_mailchimp_discount');
          break;

          case 'pending':
            $result['status'] = 'pending';
            $result['type']   = 'success';
            $result['message'] = __('A confirmation email has been sent to ', 'wc_mailchimp_discount') . $email . '. '. __('Please verify to proceed.', 'wc_mailchimp_discount');
          break;

          default:
            $result['status'] = 'error';
            $result['type']   = 'error';
          break;
        }


        if( $result['status'] == 'success' 
          && isset( $result['title'] ) 
          && $result['title'] == 'Invalid Resource' ) {
          $result['status'] = 'error';
          $result['type']   = 'error';
          $result['message'] = $result['detail'];
        }

      }
    }
    else {
      $result['status'] = 'error';
      $result['title'] = __( 'Please setup your mailchimp api key and list id.', 'wc_mailchimp_discount' );
    }
    return $result;
  }


  /**
  *
  * Our own custom method to verify the coupon for specific email address
  * as the one with woocommerce core doesn't work always.
  *
  * @param $valid boolean validation status.
  * @param $item list of values for the submitted coupon
  *
  * @return boolean status for coupon validation.
  *
  */
  public function validate_coupon( $valid, $item ) {
      
      $coupon_data = new WC_Coupon( $item );

      if( is_array( $coupon_data->get_email_restrictions() ) 
        && sizeof( $coupon_data->get_email_restrictions() ) > 0 ) {
          
          global $current_user;
          wp_get_current_user();

          $assigned_emails = $coupon_data->get_email_restrictions();
          $customer_email = $assigned_emails[0];

          if( !is_user_logged_in() 
          && $customer_email != '' 
          && strcasecmp( $customer_email, $current_user->user_email ) !== 0 ) {
            add_filter( 'woocommerce_coupon_error', array( $this,'custom_error' ), 10, 3);
            return false;
          }
          else {
            if( $customer_email != '' 
              && strcasecmp( $customer_email, $current_user->user_email ) !== 0 ) {
                add_filter( 'woocommerce_coupon_error', array( $this,'custom_error' ), 10, 3 );
              return false;
            }
          }
      }
      return $valid;
    }

  /**
  *
  * Custom error message for coupon validation.
  *
  * @param string $err default error message.
  * @param string $errcode error code for the error
  * @param array of values for the applied coupon
  *
  * @return string error message.
  *
  */
  public function custom_error( $err, $errcode, $val ) {
    if( !is_user_logged_in() )
      return __( 'Please login to apply this coupon.', 'wc_mailchimp_discount' );
    else
      return __( 'This coupon is assigned to some other user, Please verify !', 'wc_mailchimp_discount' );
  }

  //shortcode tag for generating 
  public function wcmd_render_shortcode( $atts ) {
    $options = shortcode_atts( array(
      'width'           => '100%',
      'align'           => '',
      'btn_width'       => 'auto',
      'btn_align'       => 'center',
      'top_text'        => '',
      'top_text_color'  => '#000',
      'layout'          => 'vertical'
        ), $atts );
    extract( $options );

    $addon_extra_field_id = '';
    $addon_extra_field_options = $this->option('wmcd_addon_fields');

    if( $align == 'center' )
      $align = 'margin:0 auto;';
    else if( $align == 'left' || $align == 'right' )
      $align = 'float:' . $align . ';';

    $fields = $this->option( 'wcmd_fields' );
    $form = '<div class="wcmd-form-wrapper wcmd-' . $layout . '" style="width:' . $width . '; ' . $align . '">';
    $form .= '<div class="wcmd-loading"></div>';
    
    if( $top_text != '' )
      $form .= '<div class="wcmd-top-title" style="color:' . $top_text_color . '">' . $top_text . '</div>';
        
      $form .= '<div class="wcmd_content 123">';
      $form .= '<div class="wcmd_text">';
      $form .= '<form class="wcmd-form wcmd_' . $fields . '">';
      $form .= '<div class="validation-wrap"><span class="wcmd-validation"></span></div><div class="wcmd-fields">';
      
      if( $fields == 'email_name' || $fields == 'email_name_all' )
        $form .= '<input type="text" aria-label="wcmd_fname" placeholder="'. __('Enter first name', 'wc_mailchimp_discount' ) .'" name="wcmd_fname" class="wcmd_fname">';

      if( $fields == 'email_name_all' )
        $form .= '<input type="text" aria-label="wcmd_lname" placeholder="'. __('Enter last name', 'wc_mailchimp_discount' ) .'" name="wcmd_lname" class="wcmd_lname">';
        
      $form .= '<input type="text" aria-label="wcmd_email" placeholder="'. __('Enter your email', 'wc_mailchimp_discount' ) .'" name="wcmd_email" class="wcmd_email">';

      $form .= apply_filters('wmcd_addon_fields_hook', $addon_extra_field_id, $addon_extra_field_options);

      
      //checkbox for terms and conditions
      if( $this->option('wcmd_terms_condition') == 'yes' ) :

        $term_condition_text = !empty($this->option('wcmd_terms_condition_text')) ? $this->option('wcmd_terms_condition_text') : '';
        $uniq_id = uniqid();

        $form .= '<div class="wcmd-checkbox-wrap">';
        $form .= '<input type="checkbox" id='.$uniq_id.' class="wcmd-terms-conditions" name="wcmd_terms_condition">';
        $form .= '<label for='.$uniq_id.'>'.$term_condition_text.'</label>';
        $form .= '</div>';
        
      endif;

      $form .= '</div><div class="wcmd-btn-cont" style="text-align:' . $btn_align . '">';
      $form .= '<button class="wcmd-btn" style="width:' . $btn_width . '">' . $this->option( 'wcmd_btn_text' ) . '</button>';
      $form .= '</div><div class="wcmd-clear"></div></form>';
      $form .= '<div class="wcmd-clear"></div>';
      $form .='</div></div></div>';

    return $form;
  }

}